package ui;

import java.io.*;
import java.nio.file.*;
import java.util.*;
import java.util.regex.*;
import java.security.SecureRandom;

public class FullJavaObfuscator {
    private static final SecureRandom random = new SecureRandom();
    private static final String CHAR_SET = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ";
    private static final int MIN_NAME_LENGTH = 12;
    private static final int MAX_NAME_LENGTH = 36;

    private static Map<String, String> packageMappings = new HashMap<>();
    private static Map<String, String> classMappings = new HashMap<>();
    private static Map<String, String> methodMappings = new HashMap<>();
    private static Map<String, String> variableMappings = new HashMap<>();
    private static Map<String, String> fileMappings = new HashMap<>();
    private static Path inputDir;
    private static Path outputDir;

    public FullJavaObfuscator(String put_norm, String put_obf) throws IOException {
        inputDir = Paths.get(put_norm).toAbsolutePath();
        outputDir = Paths.get(put_obf).toAbsolutePath();

        if (!Files.exists(inputDir)) {
            System.out.println("Input directory does not exist");
            return;
        }

        // Сначала сканируем все пакеты
        scanPackagesRecursively(inputDir);

        // Затем сканируем все Java файлы рекурсивно
        scanJavaFilesRecursively(inputDir);

        // Обрабатываем и переименовываем файлы
        processFilesRecursively(inputDir, outputDir);

        // Сохраняем маппинги для отладки
        saveMappings(outputDir);

        System.out.println("Obfuscation completed successfully");
    }

    private static void scanPackagesRecursively(Path dir) throws IOException {
        Files.walk(dir)
                .filter(Files::isDirectory)
                .filter(FullJavaObfuscator::containsJavaFiles)
                .forEach(path -> {
                    try {
                        String packagePath = inputDir.relativize(path).toString()
                                .replace(File.separator, ".");

                        if (!packagePath.isEmpty() && !packageMappings.containsKey(packagePath)) {
                            String newPackagePath = generatePackageName();
                            packageMappings.put(packagePath, newPackagePath);

                            // Создаем новую структуру директорий
                            Path newPackageDir = outputDir.resolve(newPackagePath.replace(".", File.separator));
                            Files.createDirectories(newPackageDir);
                        }
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                });
    }

    private static boolean containsJavaFiles(Path path) {
        try {
            return Files.list(path)
                    .anyMatch(p -> p.toString().endsWith(".java"));
        } catch (IOException e) {
            e.printStackTrace();
            return false;
        }
    }

    private static String generatePackageName() {
        int depth = 1 + random.nextInt(3); // 1-3 уровня вложенности
        StringBuilder sb = new StringBuilder();

        for (int i = 0; i < depth; i++) {
            if (i > 0) sb.append(".");
            sb.append(generateRandomName("pkg"));
        }

        return sb.toString();
    }

    private static void scanJavaFilesRecursively(Path dir) throws IOException {
        Files.walk(dir)
                .filter(path -> path.toString().endsWith(".java"))
                .forEach(path -> {
                    try {
                        String content = new String(Files.readAllBytes(path));
                        String packageName = extractPackageName(content);
                        String className = extractClassName(content);

                        // Генерируем новое имя для класса
                        String newClassName = generateRandomName("cls");
                        classMappings.put(className, newClassName);

                        // Сохраняем полное имя класса (с пакетом) для обработки импортов
                        String fullClassName = packageName.isEmpty() ? className : packageName + "." + className;
                        String newFullClassName = packageMappings.getOrDefault(packageName, packageName) + "." + newClassName;
                        classMappings.put(fullClassName, newFullClassName);

                        // Сохраняем маппинг файлов с относительными путями
                        String relativePath = inputDir.relativize(path).toString();
                        String newRelativePath = calculateNewPath(relativePath, packageName, newClassName);
                        fileMappings.put(relativePath, newRelativePath);

                        // Сканируем методы и переменные, пропуская аннотации
                        scanMethodsAndVariables(content);
                    } catch (IOException e) {
                        e.printStackTrace();
                    }
                });
    }

    private static String calculateNewPath(String originalPath, String packageName, String newClassName) {
        if (!packageMappings.containsKey(packageName)) {
            return originalPath;
        }

        String newPackagePath = packageMappings.get(packageName).replace(".", File.separator);
        return newPackagePath + File.separator + newClassName + ".java";
    }

    private static void scanMethodsAndVariables(String content) {
        // Поиск объявлений методов (пропускаем аннотации)
        Matcher methodMatcher = Pattern.compile(
                "(?<!@)\\b(?:public|protected|private|static|final|\\s)+[\\w<>\\[\\]]+\\s+(\\w+)\\s*\\([^)]*\\)")
                .matcher(content);

        while (methodMatcher.find()) {
            String methodName = methodMatcher.group(1);
            if (!methodMappings.containsKey(methodName) && !methodName.startsWith("@")) {
                methodMappings.put(methodName, generateRandomName("mth"));
            }
        }

        // Поиск объявлений переменных (пропускаем аннотации)
        Matcher varMatcher = Pattern.compile(
                "(?<!@)\\b(?:public|protected|private|static|final|\\s)+[\\w<>\\[\\]]+\\s+(\\w+)\\s*(?:=|;|\\))")
                .matcher(content);

        while (varMatcher.find()) {
            String varName = varMatcher.group(1);
            if (!variableMappings.containsKey(varName) && !varName.startsWith("@")) {
                variableMappings.put(varName, generateRandomName("var"));
            }
        }
    }

    private static void processFilesRecursively(Path inputDir, Path outputDir) throws IOException {
        if (!Files.exists(outputDir)) {
            Files.createDirectories(outputDir);
        }

        for (Map.Entry<String, String> entry : fileMappings.entrySet()) {
            Path inputFile = inputDir.resolve(entry.getKey());
            Path outputFile = outputDir.resolve(entry.getValue());

            // Создаем директории если нужно
            Files.createDirectories(outputFile.getParent());

            String content = new String(Files.readAllBytes(inputFile));
            String obfuscatedContent = obfuscateContent(content, inputDir.relativize(inputFile).toString());

            Files.write(outputFile, obfuscatedContent.getBytes());

            // Удаляем оригинальный файл, если он в выходной директории
            if (inputFile.startsWith(outputDir)) {
                Files.deleteIfExists(inputFile);
            }
        }
    }

    private static String obfuscateContent(String original, String relativePath) {
        String result = original;
        String packageName = extractPackageName(original);

        // Обрабатываем package declaration
        if (!packageName.isEmpty()) {
            String newPackage = packageMappings.getOrDefault(packageName, packageName);
            result = result.replaceFirst("package\\s+" + packageName.replace(".", "\\.") + "\\s*;",
                    "package " + newPackage + ";");
        }

        // Обрабатываем imports
        Matcher importMatcher = Pattern.compile("import\\s+((?:static\\s+)?([\\w.]+)\\.(\\w+))\\s*;").matcher(result);
        StringBuffer importBuffer = new StringBuffer();

        while (importMatcher.find()) {
            String fullImport = importMatcher.group(1);
            String importPackage = importMatcher.group(2);
            String importClass = importMatcher.group(3);

            // Для импортов из нашего проекта применяем маппинг
            if (classMappings.containsKey(importPackage + "." + importClass)) {
                String newFullImport = classMappings.get(importPackage + "." + importClass);
                importMatcher.appendReplacement(importBuffer, "import " + newFullImport + ";");
            } else if (classMappings.containsKey(importClass)) {
                // Для классов без пакета (java.lang и т.д.)
                String newImportClass = classMappings.getOrDefault(importClass, importClass);
                importMatcher.appendReplacement(importBuffer, "import " + importPackage + "." + newImportClass + ";");
            } else {
                // Оставляем как есть для системных классов
                importMatcher.appendReplacement(importBuffer, importMatcher.group(0));
            }
        }
        importMatcher.appendTail(importBuffer);
        result = importBuffer.toString();

        // Заменяем полные имена классов в коде
        for (Map.Entry<String, String> entry : classMappings.entrySet()) {
            if (entry.getKey().contains(".")) {
                result = result.replaceAll("(?<!//|import |package )\\b" + entry.getKey().replace(".", "\\.") + "\\b",
                        entry.getValue());
            }
        }

        // Заменяем простые имена классов (без пакета)
        for (Map.Entry<String, String> entry : classMappings.entrySet()) {
            if (!entry.getKey().contains(".")) {
                result = result.replaceAll("(?<!//|import |package |\\.)\\b" + entry.getKey() + "\\b",
                        entry.getValue().substring(entry.getValue().lastIndexOf(".") + 1));
            }
        }

        // Заменяем имена методов (но не в аннотациях)
        for (Map.Entry<String, String> entry : methodMappings.entrySet()) {
            result = result.replaceAll("(?<!@)\\b" + entry.getKey() + "\\(", entry.getValue() + "(");
        }

        // Заменяем имена переменных (но не в аннотациях)
        for (Map.Entry<String, String> entry : variableMappings.entrySet()) {
            result = result.replaceAll("(?<!@)\\b" + entry.getKey() + "\\b", entry.getValue());
        }

        return result;
    }

    private static String extractPackageName(String content) {
        Matcher matcher = Pattern.compile("package\\s+([\\w.]+)\\s*;").matcher(content);
        return matcher.find() ? matcher.group(1) : "";
    }

    private static String extractClassName(String content) {
        Matcher matcher = Pattern.compile(
                "\\b(?:public|protected|private|static|final|abstract|\\s)*class\\s+(\\w+)")
                .matcher(content);
        if (matcher.find()) {
            return matcher.group(1);
        }
        throw new RuntimeException("Class name not found in: " + content);
    }

    private static String generateRandomName(String prefix) {
        int length = MIN_NAME_LENGTH + random.nextInt(MAX_NAME_LENGTH - MIN_NAME_LENGTH + 1);
        StringBuilder sb = new StringBuilder(prefix);

        for (int i = 0; i < length; i++) {
            sb.append(CHAR_SET.charAt(random.nextInt(CHAR_SET.length())));
        }

        return sb.toString();
    }

    private static void saveMappings(Path outputDir) throws IOException {
        Path mappingsFile = outputDir.resolve("obfuscation_mappings.txt");

        try (PrintWriter writer = new PrintWriter(Files.newBufferedWriter(mappingsFile))) {
            writer.println("=== Package Mappings ===");
            packageMappings.entrySet().stream()
                    .sorted(Map.Entry.comparingByKey())
                    .forEach(entry -> writer.println(entry.getKey() + " -> " + entry.getValue()));

            writer.println("\n=== Class Mappings ===");
            classMappings.entrySet().stream()
                    .filter(e -> !e.getKey().contains("."))
                    .sorted(Map.Entry.comparingByKey())
                    .forEach(entry -> writer.println(entry.getKey() + " -> " + entry.getValue()));

            writer.println("\n=== Full Class Name Mappings ===");
            classMappings.entrySet().stream()
                    .filter(e -> e.getKey().contains("."))
                    .sorted(Map.Entry.comparingByKey())
                    .forEach(entry -> writer.println(entry.getKey() + " -> " + entry.getValue()));

            writer.println("\n=== Method Mappings ===");
            methodMappings.entrySet().stream()
                    .sorted(Map.Entry.comparingByKey())
                    .forEach(entry -> writer.println(entry.getKey() + " -> " + entry.getValue()));

            writer.println("\n=== Variable Mappings ===");
            variableMappings.entrySet().stream()
                    .sorted(Map.Entry.comparingByKey())
                    .forEach(entry -> writer.println(entry.getKey() + " -> " + entry.getValue()));

            writer.println("\n=== File Mappings ===");
            fileMappings.entrySet().stream()
                    .sorted(Map.Entry.comparingByKey())
                    .forEach(entry -> writer.println(entry.getKey() + " -> " + entry.getValue()));
        }
    }
}